const express = require('express');
const router = express.Router();
const { body } = require('express-validator');
const authController = require('../controllers/authController');
const { validate } = require('../middleware/validator');
const { authenticate } = require('../middleware/auth');

// Validation rules
const registerValidation = [
  body('email').isEmail().normalizeEmail().withMessage('Valid email is required'),
  body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
  body('role').isIn(['client', 'detailer']).withMessage('Role must be client or detailer'),
  body('phone').matches(/^[0-9+\-\s()]+$/).withMessage('Valid phone number is required'),
  
  // Client-specific validation
  body('first_name').if(body('role').equals('client')).trim().notEmpty().withMessage('First name is required for clients'),
  body('last_name').if(body('role').equals('client')).trim().notEmpty().withMessage('Last name is required for clients'),
  
  // Detailer-specific validation
  body('business_name').if(body('role').equals('detailer')).trim().notEmpty().withMessage('Business name is required for detailers'),
  body('owner_name').if(body('role').equals('detailer')).trim().notEmpty().withMessage('Owner name is required for detailers'),
  body('phone_verification_id').isInt({ min: 1 }).withMessage('Phone verification is required'),
];

const loginValidation = [
  body('identifier').optional().trim(),
  body('email').optional().trim(),
  body('password').notEmpty().withMessage('Password is required'),
  body().custom((value) => {
    const identifier = String(value?.identifier || value?.email || '').trim();
    if (!identifier) {
      throw new Error('Email or phone is required');
    }
    return true;
  }),
];

const forgotPasswordValidation = [
  body('channel').isIn(['email', 'sms']).withMessage('Channel must be email or sms'),
  body('identifier').notEmpty().withMessage('Email or phone is required')
];

const verifyOtpValidation = [
  body('channel').isIn(['email', 'sms']).withMessage('Channel must be email or sms'),
  body('identifier').notEmpty().withMessage('Email or phone is required'),
  body('otp').isLength({ min: 4 }).withMessage('OTP is required')
];

const resetPasswordValidation = [
  body('resetToken').notEmpty().withMessage('Reset token is required'),
  body('newPassword').isLength({ min: 6 }).withMessage('Password must be at least 6 characters')
];

// Routes
router.post('/register', registerValidation, validate, authController.register);
router.post('/check-availability', authController.checkAvailability);
router.post('/send-registration-otp', authController.sendRegistrationOtp);
router.post('/verify-registration-otp', authController.verifyRegistrationOtp);
router.post('/login', loginValidation, validate, authController.login);
router.post('/forgot-password', forgotPasswordValidation, validate, authController.forgotPassword);
router.post('/verify-otp', verifyOtpValidation, validate, authController.verifyOtp);
router.post('/reset-password', resetPasswordValidation, validate, authController.resetPassword);
router.post('/refresh-token', authController.refreshToken);
router.get('/me', authenticate, authController.getProfile);
router.put('/change-password', authenticate, authController.changePassword);
router.post('/logout', authenticate, authController.logout);

module.exports = router;
