const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');
const { initializePayment, handleCallback, verifyPayment } = require('../services/paymentService');
const { sendPushToUser } = require('../services/notificationService');

/**
 * @desc    Initialize PayChangu payment
 * @route   POST /api/v1/payments/initialize
 * @access  Private (Client only)
 */
exports.initializePayChangu = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { bookingId } = req.body;

  if (!bookingId) {
    return res.status(400).json({
      success: false,
      message: 'Booking ID is required',
    });
  }

  try {
    // Get booking details - must be completed status for payment
    const [bookings] = await db.query(
      `SELECT b.*, s.price, c.phone as client_phone, c.first_name, c.last_name, u.email as user_email
       FROM bookings b
       JOIN services s ON b.service_id = s.id
       JOIN clients c ON b.client_id = c.id
       JOIN users u ON c.user_id = u.id
       WHERE b.id = ? AND b.client_id = (SELECT id FROM clients WHERE user_id = ?)`,
      [bookingId, userId]
    );

    if (bookings.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Booking not found',
      });
    }

    const booking = bookings[0];

    // Check if booking is completed (ready for payment)
    if (booking.status !== 'completed') {
      return res.status(400).json({
        success: false,
        message: 'Payment can only be made for completed services',
      });
    }

    // Check if already paid
    const [existingPayments] = await db.query(
      'SELECT id, status FROM payments WHERE booking_id = ? AND status IN (\"verified\", \"pending\")',
      [bookingId]
    );

    if (existingPayments.length > 0) {
      return res.status(400).json({
        success: false,
        message: existingPayments[0].status === 'verified' 
          ? 'This booking has already been paid' 
          : 'A payment for this booking is already in progress',
      });
    }

    // Initialize payment with PayChangu
    const paymentInit = await initializePayment({
      bookingId: booking.id,
      amount: booking.amount,
      description: `Booking #${booking.booking_number} - Car Wash Service`,
      customerEmail: booking.user_email,
      customerFirstName: booking.first_name,
      customerLastName: booking.last_name,
      customerPhone: booking.client_phone,
      returnUrl: process.env.PAYCHANGU_RETURN_URL || 'simbi://payment-return',
      callbackUrl: process.env.PAYCHANGU_CALLBACK_URL || `${process.env.BACKEND_URL || 'http://localhost:5000'}/api/v1/payments/callback`,
    });

    const tx_ref = paymentInit.data.tx_ref;

    // Create a pending payment record to establish tx_ref -> bookingId mapping
    // This allows us to find the booking when the webhook callback arrives
    await db.query(
      `INSERT INTO payments (booking_id, tx_ref, amount, payment_method, status, gateway, created_at)
       VALUES (?, ?, ?, ?, ?, ?, NOW())
       ON DUPLICATE KEY UPDATE updated_at = NOW()`,
      [booking.id, tx_ref, booking.amount, 'card', 'pending', 'paychangu']
    );

    console.log('[PAYMENT] Payment initialized for booking:', booking.id, 'tx_ref:', tx_ref);

    res.json({
      success: true,
      message: 'Payment initialized successfully',
      data: {
        bookingId: booking.id,
        bookingNumber: booking.booking_number,
        amount: booking.amount,
        ...paymentInit.data,
      },
    });
  } catch (error) {
    console.error('[PAYMENT] Payment initialization error:', error);
    throw error;
  }
});

/**
 * @desc    Handle PayChangu payment callback (webhook)
 * @route   POST /api/v1/payments/callback
 * @access  Public (PayChangu webhook with signature verification)
 */
exports.paymentCallback = asyncHandler(async (req, res) => {
  const callbackData = req.body;
  const signature = req.headers['x-paychangu-signature'];

  // Verify webhook signature for security
  if (!signature) {
    console.warn('[PAYMENT] Webhook rejected: Missing signature', {
      ip: req.ip,
      timestamp: new Date().toISOString(),
    });
    return res.status(401).json({
      success: false,
      message: 'Unauthorized: Missing signature',
    });
  }

  // Verify signature
  const { verifySignature } = require('../services/paymentService');
  if (!verifySignature(callbackData, signature)) {
    console.warn('[PAYMENT] Webhook rejected: Invalid signature', {
      ip: req.ip,
      tx_ref: callbackData.tx_ref,
      timestamp: new Date().toISOString(),
    });
    return res.status(401).json({
      success: false,
      message: 'Unauthorized: Invalid signature',
    });
  }

  console.log('[PAYMENT] Received valid payment callback:', {
    tx_ref: callbackData.tx_ref,
    transaction_id: callbackData.transaction_id,
    status: callbackData.status,
  });

  try {
    // Process callback with signature and database connection
    const result = await handleCallback(callbackData, signature, db);

    if (!result.success) {
      return res.status(400).json({
        success: false,
        message: 'Invalid callback',
      });
    }

    const { bookingId, status, transactionId, tx_ref } = result;

    // Verify payment on PayChangu server using tx_ref (not transactionId)
    const verification = await verifyPayment(tx_ref);

    if (!verification.verified) {
      console.error('[PAYMENT] Payment verification failed for tx_ref:', tx_ref);
      // Return 202 to tell PayChangu to retry
      return res.status(202).json({
        success: false,
        message: 'Payment verification in progress, will retry',
      });
    }

    const connection = await db.getConnection();

    try {
      await connection.beginTransaction();

      // VERIFY: Booking exists and is in correct state
      const [bookings] = await connection.query(
        `SELECT id, amount, status FROM bookings WHERE id = ? AND deleted_at IS NULL`,
        [bookingId]
      );

      if (bookings.length === 0) {
        await connection.rollback();
        console.warn('[PAYMENT] Booking not found for callback:', { bookingId });
        return res.status(404).json({
          success: false,
          message: 'Booking not found',
        });
      }

      const booking = bookings[0];

      // VERIFY: Amount matches exactly
      if (parseFloat(booking.amount) !== parseFloat(verification.amount)) {
        await connection.rollback();
        console.error('[PAYMENT] Amount mismatch:', {
          bookingId,
          expectedAmount: booking.amount,
          receivedAmount: verification.amount,
        });
        return res.status(400).json({
          success: false,
          message: 'Payment amount mismatch',
        });
      }

      // VERIFY: Booking is in pending status
      if (booking.status !== 'pending') {
        await connection.rollback();
        console.warn('[PAYMENT] Booking not in pending status:', {
          bookingId,
          status: booking.status,
        });
        return res.status(400).json({
          success: false,
          message: 'Booking not in pending status',
        });
      }

      // Check if payment already exists by tx_ref (idempotency check)
      const [existingPayments] = await connection.query(
        'SELECT id, status FROM payments WHERE tx_ref = ?',
        [tx_ref]
      );

      if (existingPayments.length > 0 && existingPayments[0].status === 'verified') {
        await connection.commit();
        console.log('[PAYMENT] Payment already verified (idempotent return):', {
          transactionId,
          tx_ref: tx_ref,
        });
        return res.json({
          success: true,
          message: 'Payment already processed',
          data: {
            bookingId,
            transactionId,
            amount: verification.amount,
          },
        });
      }

      // Update existing pending payment record (don't create duplicate)
      await connection.query(
        `UPDATE payments 
         SET status = ?, transaction_id = ?, payment_date = NOW(), updated_at = NOW()
         WHERE tx_ref = ?`,
        ['verified', transactionId, tx_ref]
      );

      // Update booking status to paid (service completed and paid)
      await connection.query(
        'UPDATE bookings SET status = ? WHERE id = ?',
        ['paid', bookingId]
      );

      // Create notification for admin/detailer about payment
      const [bookingDetails] = await connection.query(
        `SELECT b.*, d.user_id as detailer_user_id 
         FROM bookings b
         JOIN detailers d ON b.detailer_id = d.id
         WHERE b.id = ?`,
        [bookingId]
      );

      if (bookingDetails.length > 0) {
        const booking = bookingDetails[0];

        // Notify detailer about payment
        await connection.query(
          `INSERT INTO notifications (user_id, type, title, message, data)
           VALUES (?, 'payment_received', 'Payment Received', ?, ?)`,
          [
            booking.detailer_user_id,
            `Payment received for booking ${booking.booking_number}.`,
            JSON.stringify({ booking_id: bookingId }),
          ]
        );

        try {
          await sendPushToUser({
            userId: booking.detailer_user_id,
            type: 'payment_received',
            title: 'Payment Received',
            message: `Payment received for booking ${booking.booking_number}.`,
            data: { booking_id: bookingId },
          });
        } catch (pushError) {
          console.error('[PUSH] Failed to send payment_received push:', pushError.message);
        }
      }

      await connection.commit();

      console.log('[PAYMENT] Payment processed successfully:', {
        bookingId,
        transactionId,
        amount: verification.amount,
      });

      res.json({
        success: true,
        message: 'Payment processed successfully',
        data: {
          bookingId,
          transactionId,
          amount: verification.amount,
        },
      });
    } catch (error) {
      await connection.rollback();
      throw error;
    } finally {
      connection.release();
    }
  } catch (error) {
    console.error('[PAYMENT] Callback processing error:', error.message, {
      tx_ref: callbackData.tx_ref,
      transaction_id: callbackData.transaction_id,
      stack: error.stack,
    });
    
    // Return 202 on error to tell PayChangu to retry
    res.status(202).json({
      success: false,
      message: 'Payment processing in progress, will retry',
    });
  }
});

/**
 * @desc    Handle PayChangu payment return
 * @route   GET /api/v1/payments/return
 * @access  Public (Called by PayChangu after payment)
 */
exports.paymentReturn = asyncHandler(async (req, res) => {
  const { tx_ref, status, transaction_id } = req.query;

  console.log('[PAYMENT] Return URL called:', {
    tx_ref,
    status,
    transaction_id,
    query: req.query,
  });

  // Trigger payment verification and database insertion in the background
  if ((status === 'success' || status === 'successful') && tx_ref) {
    // Process payment asynchronously - don't wait for it
    setImmediate(async () => {
      let connection;
      try {
        connection = await db.getConnection();
        await connection.beginTransaction();

        // Check if payment already exists
        const [existingPayments] = await connection.query(
          'SELECT id FROM payments WHERE tx_ref = ?',
          [tx_ref]
        );

        if (existingPayments.length > 0) {
          console.log('[PAYMENT] Payment already processed for tx_ref:', tx_ref);
          await connection.rollback();
          connection.release();
          return;
        }

        // Verify payment with PayChangu
        const verification = await verifyPayment(tx_ref);
        
        if (!verification.verified) {
          console.log('[PAYMENT] Payment verification failed for tx_ref:', tx_ref);
          await connection.rollback();
          connection.release();
          return;
        }

        console.log('[PAYMENT] Payment verified from PayChangu:', verification);

        // Find booking by tx_ref from the pending payment record
        const [payments] = await connection.query(
          `SELECT p.booking_id, b.booking_number, b.amount, b.status, d.user_id as detailer_user_id
           FROM payments p
           JOIN bookings b ON p.booking_id = b.id
           LEFT JOIN detailers d ON b.detailer_id = d.id
           WHERE p.tx_ref = ?`,
          [tx_ref]
        );

        if (payments.length === 0) {
          console.log('[PAYMENT] No payment found for tx_ref:', tx_ref);
          await connection.rollback();
          connection.release();
          return;
        }

        const booking = payments[0];
        const bookingId = booking.booking_id;

        console.log('[PAYMENT] Found booking for payment:', {
          bookingId,
          bookingNumber: booking.booking_number,
          tx_ref,
        });

        // Use verified amount from PayChangu, fallback to booking amount
        const paymentAmount = verification.amount || booking.amount;

        // Update existing payment record to verified
        await connection.query(
          `UPDATE payments 
           SET status = ?, transaction_id = ?, payment_date = NOW(), updated_at = NOW()
           WHERE tx_ref = ?`,
          ['verified', verification.transactionId, tx_ref]
        );

        console.log('[PAYMENT] Payment record updated to verified for tx_ref:', tx_ref);

        // Update booking status to paid
        await connection.query(
          'UPDATE bookings SET status = ? WHERE id = ?',
          ['paid', bookingId]
        );

        // Create notification for detailer about payment
        if (booking.detailer_user_id) {
          await connection.query(
            `INSERT INTO notifications (user_id, type, title, message, data, created_at)
             VALUES (?, 'payment_received', 'Payment Received', ?, ?, NOW())`,
            [
              booking.detailer_user_id,
              `Payment received for booking ${booking.booking_number}.`,
              JSON.stringify({ booking_id: bookingId }),
            ]
          );

          try {
            await sendPushToUser({
              userId: booking.detailer_user_id,
              type: 'payment_received',
              title: 'Payment Received',
              message: `Payment received for booking ${booking.booking_number}.`,
              data: { booking_id: bookingId },
            });
          } catch (pushError) {
            console.error('[PUSH] Failed to send payment_received push:', pushError.message);
          }
        }

        await connection.commit();
        console.log('[PAYMENT] ✓ Payment processed successfully from return URL:', {
          bookingId,
          tx_ref,
          transactionId: verification.transactionId,
          amount: paymentAmount,
        });
      } catch (error) {
        if (connection) {
          try {
            await connection.rollback();
          } catch (rollbackError) {
            console.error('[PAYMENT] Rollback error:', rollbackError.message);
          }
        }
        console.error('[PAYMENT] ✗ Error processing payment from return URL:', error.message, {
          tx_ref,
          stack: error.stack,
        });
      } finally {
        if (connection) {
          connection.release();
        }
      }
    });
  }
  
  // Send a simple success page that closes itself
  res.send(`
    <!DOCTYPE html>
    <html>
      <head>
        <meta charset="utf-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1.0" />
        <title>Payment ${status === 'success' || status === 'successful' ? 'Successful' : 'Processing'}</title>
        <style>
          body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            margin: 0;
            background: #f3f4f6;
          }
          .container {
            text-align: center;
            padding: 40px;
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            max-width: 400px;
          }
          .icon {
            font-size: 64px;
            margin-bottom: 20px;
          }
          .success { color: #10b981; }
          .processing { color: #f59e0b; }
          h1 { color: #1f2937; margin: 0 0 16px; font-size: 24px; }
          p { color: #6b7280; margin: 0 0 24px; }
          .close-msg { font-size: 14px; color: #9ca3af; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="icon ${status === 'success' || status === 'successful' ? 'success' : 'processing'}">
            ${status === 'success' || status === 'successful' ? '✓' : '⏳'}
          </div>
          <h1>${status === 'success' || status === 'successful' ? 'Payment Successful!' : 'Payment Processing'}</h1>
          <p>${status === 'success' || status === 'successful' 
            ? 'Your payment has been received and is being verified.' 
            : 'Your payment is being processed. Please wait...'}</p>
          <p class="close-msg">You can close this page and return to the app.</p>
        </div>
        <script>
          // Auto-close after 3 seconds
          setTimeout(function() {
            window.close();
          }, 3000);
        </script>
      </body>
    </html>
  `);
});

/**
 * @desc    Verify PayChangu payment status
 * @route   POST /api/v1/payments/verify-status
 * @access  Private (Client only)
 */
exports.verifyPaymentStatus = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { bookingId } = req.body;

  if (!bookingId) {
    return res.status(400).json({
      success: false,
      message: 'Booking ID is required',
    });
  }

  try {
    // Check payment status
    const [payments] = await db.query(
      `SELECT p.* FROM payments p
       JOIN bookings b ON p.booking_id = b.id
       JOIN clients c ON b.client_id = c.id
       WHERE b.id = ? AND c.user_id = ?
       ORDER BY p.created_at DESC
       LIMIT 1`,
      [bookingId, userId]
    );

    if (payments.length === 0) {
      return res.status(200).json({
        success: true,
        data: {
          status: 'not_found',
          message: 'No payment found yet',
        },
      });
    }

    const payment = payments[0];

    // If payment is verified, get booking details
    let bookingStatus = null;
    if (payment.status === 'verified') {
      const [bookingData] = await db.query(
        'SELECT id, status FROM bookings WHERE id = ?',
        [bookingId]
      );
      if (bookingData.length > 0) {
        bookingStatus = bookingData[0].status;
      }
    }

    res.json({
      success: true,
      data: {
        paymentId: payment.id,
        status: payment.status,
        amount: payment.amount,
        transactionId: payment.transaction_id,
        paymentDate: payment.payment_date,
        bookingStatus,
      },
    });
  } catch (error) {
    console.error('[PAYMENT] Payment verification error:', error);
    throw error;
  }
});

/**
 * @desc    Poll payment status (for clients to check status without deep link)
 * @route   POST /api/v1/payments/poll-status
 * @access  Private (Client only)
 */
exports.pollPaymentStatus = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { bookingId } = req.body;

  if (!bookingId) {
    return res.status(400).json({
      success: false,
      message: 'Booking ID is required',
    });
  }

  try {
    // Verify booking belongs to user
    const [access] = await db.query(
      `SELECT b.id FROM bookings b
       JOIN clients c ON b.client_id = c.id
       WHERE b.id = ? AND c.user_id = ? AND b.deleted_at IS NULL`,
      [bookingId, userId]
    );

    if (access.length === 0) {
      return res.status(403).json({
        success: false,
        message: 'Access denied',
      });
    }

    // Get latest payment
    const [payments] = await db.query(
      `SELECT id, status, amount, transaction_id, tx_ref, gateway, payment_date, created_at
       FROM payments
       WHERE booking_id = ?
       ORDER BY created_at DESC
       LIMIT 1`,
      [bookingId]
    );

    if (payments.length === 0) {
      return res.json({
        success: true,
        found: false,
        message: 'No payment found',
      });
    }

    const payment = payments[0];

    // If pending and using PayChangu, don't verify during polling
    // Let the client call /verify-by-tx-ref after payment completion
    // This avoids unnecessary API calls to PayChangu during polling
    // Only return the payment status that we have
    
    if (payment.status === 'pending' && payment.gateway === 'paychangu') {
      console.log('[PAYMENT] Payment still pending, not verifying during poll:', payment.tx_ref);
    }

    res.json({
      success: true,
      found: true,
      data: {
        id: payment.id,
        status: payment.status,
        amount: payment.amount,
        gateway: payment.gateway,
        createdAt: payment.created_at,
        isVerified: payment.status === 'verified',
      },
    });
  } catch (error) {
    console.error('[PAYMENT] Payment poll error:', error);
    throw error;
  }
});

/**
 * @desc    Verify payment by transaction reference (called by client after PayChangu popup closes)
 * @route   POST /api/v1/payments/verify-by-tx-ref
 * @access  Private (Client only)
 */
exports.verifyPaymentByTxRef = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { tx_ref } = req.body;

  if (!tx_ref) {
    return res.status(400).json({
      success: false,
      message: 'Transaction reference is required',
    });
  }

  try {
    // Find payment by tx_ref
    const [payments] = await db.query(
      `SELECT p.*, b.id as booking_id, b.status as booking_status, b.amount as booking_amount,
              c.user_id
       FROM payments p
       JOIN bookings b ON p.booking_id = b.id
       JOIN clients c ON b.client_id = c.id
       WHERE p.tx_ref = ?`,
      [tx_ref]
    );

    if (payments.length === 0) {
      console.log('[PAYMENT] No payment found for tx_ref:', tx_ref);
      return res.json({
        success: true,
        found: false,
        message: 'Payment not found or not yet processed',
      });
    }

    const payment = payments[0];

    // Verify user owns this payment
    if (payment.user_id !== userId) {
      console.warn('[PAYMENT] Unauthorized access attempt for tx_ref:', tx_ref);
      return res.status(403).json({
        success: false,
        message: 'Unauthorized',
      });
    }

    // If payment is still pending, try to verify with PayChangu
    if (payment.status === 'pending') {
      try {
        const verification = await verifyPayment(tx_ref);
        
        if (verification.verified) {
          // Update payment to verified
          const connection = await db.getConnection();
          try {
            await connection.beginTransaction();

            // Update payment status
            await connection.query(
              'UPDATE payments SET status = ?, transaction_id = ?, verified_at = NOW() WHERE tx_ref = ?',
              ['verified', verification.transactionId, tx_ref]
            );

            // Update booking status to paid (completed + paid)
            await connection.query(
              'UPDATE bookings SET status = ? WHERE id = ?',
              ['paid', payment.booking_id]
            );

            // Create notification for detailer
            const [bookingData] = await connection.query(
              'SELECT detailer_id, booking_number FROM bookings WHERE id = ?',
              [payment.booking_id]
            );

            if (bookingData.length > 0 && bookingData[0].detailer_id) {
              const [detailer] = await connection.query(
                'SELECT user_id FROM detailers WHERE id = ?',
                [bookingData[0].detailer_id]
              );

              if (detailer.length > 0) {
                await connection.query(
                  `INSERT INTO notifications (user_id, type, title, message, data)
                   VALUES (?, 'payment_received', 'Payment Received', ?, ?)`,
                  [
                    detailer[0].user_id,
                    `Payment received for booking ${bookingData[0].booking_number}.`,
                    JSON.stringify({ booking_id: payment.booking_id }),
                  ]
                );

                try {
                  await sendPushToUser({
                    userId: detailer[0].user_id,
                    type: 'payment_received',
                    title: 'Payment Received',
                    message: `Payment received for booking ${bookingData[0].booking_number}.`,
                    data: { booking_id: payment.booking_id },
                  });
                } catch (pushError) {
                  console.error('[PUSH] Failed to send payment_received push:', pushError.message);
                }
              }
            }

            await connection.commit();
            console.log('[PAYMENT] ✓ Payment verified and booking updated for tx_ref:', tx_ref);
          } catch (error) {
            await connection.rollback();
            throw error;
          } finally {
            connection.release();
          }

          payment.status = 'verified';
        }
      } catch (error) {
        console.error('[PAYMENT] Error verifying payment with PayChangu:', error.message);
        // Continue - return current status
      }
    }

    res.json({
      success: true,
      found: true,
      data: {
        id: payment.id,
        tx_ref: payment.tx_ref,
        bookingId: payment.booking_id,
        status: payment.status,
        amount: payment.amount,
        bookingStatus: payment.booking_status,
        isVerified: payment.status === 'verified',
      },
    });
  } catch (error) {
    console.error('[PAYMENT] Payment verification by tx_ref error:', error);
    throw error;
  }
});