const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');

/**
 * @desc    Get active offers (public)
 * @route   GET /api/v1/offers
 * @access  Public
 */
exports.getOffers = asyncHandler(async (req, res) => {
  const [offers] = await db.query(
    `SELECT
      o.id, o.title, o.subtitle, o.description, o.image_url,
      o.discount_type, o.discount_value, o.new_users_only,
      o.starts_at, o.ends_at,
      o.service_id, o.detailer_id,
      s.name AS service_name, s.price AS service_price, s.duration_minutes, s.location_type,
      d.business_name, d.address, d.city, d.latitude, d.longitude
    FROM offers o
    INNER JOIN services s ON o.service_id = s.id
    INNER JOIN detailers d ON o.detailer_id = d.id
    WHERE o.deleted_at IS NULL
      AND o.is_active = 1
      AND (o.starts_at IS NULL OR o.starts_at = '0000-00-00 00:00:00' OR o.starts_at <= NOW())
      AND (o.ends_at IS NULL OR o.ends_at = '0000-00-00 00:00:00' OR o.ends_at >= NOW())
    ORDER BY o.created_at DESC`
  );

  res.json({ success: true, data: offers });
});

/**
 * @desc    Get offer by ID (public)
 * @route   GET /api/v1/offers/:id
 * @access  Public
 */
exports.getOfferById = asyncHandler(async (req, res) => {
  const { id } = req.params;

  const [rows] = await db.query(
    `SELECT
      o.*,
      s.name AS service_name, s.price AS service_price, s.duration_minutes, s.location_type,
      d.business_name, d.address, d.city, d.latitude, d.longitude
    FROM offers o
    INNER JOIN services s ON o.service_id = s.id
    INNER JOIN detailers d ON o.detailer_id = d.id
    WHERE o.id = ? AND o.deleted_at IS NULL
    LIMIT 1`,
    [id]
  );

  if (rows.length === 0) {
    return res.status(404).json({ success: false, message: 'Offer not found' });
  }

  res.json({ success: true, data: rows[0] });
});

/**
 * @desc    Create offer (admin)
 * @route   POST /api/v1/offers
 * @access  Private (Admin)
 */
exports.createOffer = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const {
    title,
    subtitle,
    description,
    image_url,
    service_id,
    discount_type = 'percent',
    discount_value = 0,
    new_users_only = 0,
    starts_at = null,
    ends_at = null,
    is_active = 1,
  } = req.body;

  if (!title || !service_id) {
    return res.status(400).json({ success: false, message: 'title and service_id are required' });
  }

  // derive detailer_id from service (admin "chooses service", detailer is tied)
  const [services] = await db.query(
    'SELECT id, detailer_id FROM services WHERE id = ? AND deleted_at IS NULL',
    [service_id]
  );
  if (services.length === 0) {
    return res.status(404).json({ success: false, message: 'Service not found' });
  }
  const detailer_id = services[0].detailer_id;

  const [result] = await db.query(
    `INSERT INTO offers
      (title, subtitle, description, image_url, service_id, detailer_id,
       discount_type, discount_value, new_users_only, starts_at, ends_at, is_active, created_by)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
    [
      title,
      subtitle || null,
      description || null,
      image_url || null,
      service_id,
      detailer_id,
      discount_type,
      Number(discount_value) || 0,
      new_users_only ? 1 : 0,
      starts_at,
      ends_at,
      is_active ? 1 : 0,
      userId,
    ]
  );

  res.status(201).json({ success: true, message: 'Offer created', data: { id: result.insertId } });
});

/**
 * @desc    Update offer (admin)
 * @route   PUT/PATCH /api/v1/offers/:id
 * @access  Private (Admin)
 */
exports.updateOffer = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  const allowed = [
    'title',
    'subtitle',
    'description',
    'image_url',
    'discount_type',
    'discount_value',
    'new_users_only',
    'starts_at',
    'ends_at',
    'is_active',
  ];

  const fields = [];
  const values = [];
  for (const key of allowed) {
    if (req.body[key] !== undefined) {
      fields.push(`${key} = ?`);
      if (key === 'new_users_only' || key === 'is_active') {
        values.push(req.body[key] ? 1 : 0);
      } else if (key === 'discount_value') {
        values.push(Number(req.body[key]) || 0);
      } else {
        values.push(req.body[key]);
      }
    }
  }

  if (fields.length === 0) {
    return res.status(400).json({ success: false, message: 'No fields to update' });
  }

  // keep audit of who last touched it
  fields.push('updated_at = NOW()');
  values.push(id);

  const [result] = await db.query(
    `UPDATE offers SET ${fields.join(', ')} WHERE id = ? AND deleted_at IS NULL`,
    values
  );

  if (result.affectedRows === 0) {
    return res.status(404).json({ success: false, message: 'Offer not found' });
  }

  res.json({ success: true, message: 'Offer updated' });
});

