const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');
const { isExpoPushToken } = require('../services/notificationService');

/**
 * @desc    Get user notifications
 * @route   GET /api/v1/notifications
 * @access  Private
 */
exports.getNotifications = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { page = 1, limit = 20, unread_only = false } = req.query;

  let query = `
    SELECT id, type, title, message, data, is_read, read_at, created_at
    FROM notifications
    WHERE user_id = ?
  `;

  const params = [userId];

  if (unread_only === 'true') {
    query += ' AND is_read = FALSE';
  }

  query += ' ORDER BY created_at DESC LIMIT ? OFFSET ?';
  const offset = (parseInt(page) - 1) * parseInt(limit);
  params.push(parseInt(limit), offset);

  const [notifications] = await db.query(query, params);

  // Get unread count
  const [countResult] = await db.query(
    'SELECT COUNT(*) as unread_count FROM notifications WHERE user_id = ? AND is_read = FALSE',
    [userId]
  );

  res.json({
    success: true,
    data: notifications,
    unread_count: countResult[0].unread_count
  });
});

/**
 * @desc    Mark notification as read
 * @route   PUT /api/v1/notifications/:id/read
 * @access  Private
 */
exports.markAsRead = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  const [result] = await db.query(
    'UPDATE notifications SET is_read = TRUE, read_at = NOW() WHERE id = ? AND user_id = ?',
    [id, userId]
  );

  if (result.affectedRows === 0) {
    return res.status(404).json({
      success: false,
      message: 'Notification not found'
    });
  }

  res.json({
    success: true,
    message: 'Notification marked as read'
  });
});

/**
 * @desc    Mark all notifications as read
 * @route   PUT /api/v1/notifications/read-all
 * @access  Private
 */
exports.markAllAsRead = asyncHandler(async (req, res) => {
  const userId = req.user.id;

  await db.query(
    'UPDATE notifications SET is_read = TRUE, read_at = NOW() WHERE user_id = ? AND is_read = FALSE',
    [userId]
  );

  res.json({
    success: true,
    message: 'All notifications marked as read'
  });
});

/**
 * @desc    Delete notification
 * @route   DELETE /api/v1/notifications/:id
 * @access  Private
 */
exports.deleteNotification = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  const [result] = await db.query(
    'DELETE FROM notifications WHERE id = ? AND user_id = ?',
    [id, userId]
  );

  if (result.affectedRows === 0) {
    return res.status(404).json({
      success: false,
      message: 'Notification not found'
    });
  }

  res.json({
    success: true,
    message: 'Notification deleted'
  });
});

/**
 * @desc    Register Expo push token
 * @route   POST /api/v1/notifications/push-token/register
 * @access  Private
 */
exports.registerPushToken = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { token, platform = 'unknown', device_id = null } = req.body || {};

  if (!token || !isExpoPushToken(token)) {
    return res.status(400).json({
      success: false,
      message: 'Invalid Expo push token',
    });
  }

  const normalizedPlatform = ['android', 'ios', 'web'].includes(platform) ? platform : 'unknown';

  await db.query(
    `INSERT INTO user_push_tokens (user_id, platform, expo_push_token, device_id, is_active, last_seen_at)
     VALUES (?, ?, ?, ?, 1, NOW())
     ON DUPLICATE KEY UPDATE
       platform = VALUES(platform),
       device_id = VALUES(device_id),
       is_active = 1,
       last_seen_at = NOW(),
       updated_at = CURRENT_TIMESTAMP`,
    [userId, normalizedPlatform, token, device_id]
  );

  await db.query(
    `INSERT INTO notification_preferences (user_id)
     VALUES (?)
     ON DUPLICATE KEY UPDATE updated_at = CURRENT_TIMESTAMP`,
    [userId]
  );

  res.json({
    success: true,
    message: 'Push token registered',
  });
});

/**
 * @desc    Unregister Expo push token
 * @route   POST /api/v1/notifications/push-token/unregister
 * @access  Private
 */
exports.unregisterPushToken = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { token, device_id = null } = req.body || {};

  if (!token && !device_id) {
    return res.status(400).json({
      success: false,
      message: 'Provide token or device_id',
    });
  }

  if (token) {
    await db.query(
      `UPDATE user_push_tokens
       SET is_active = 0, updated_at = CURRENT_TIMESTAMP
       WHERE user_id = ? AND expo_push_token = ?`,
      [userId, token]
    );
  } else {
    await db.query(
      `UPDATE user_push_tokens
       SET is_active = 0, updated_at = CURRENT_TIMESTAMP
       WHERE user_id = ? AND device_id = ?`,
      [userId, device_id]
    );
  }

  res.json({
    success: true,
    message: 'Push token unregistered',
  });
});

/**
 * @desc    Get notification preferences
 * @route   GET /api/v1/notifications/preferences
 * @access  Private
 */
exports.getNotificationPreferences = asyncHandler(async (req, res) => {
  const userId = req.user.id;

  const [rows] = await db.query(
    `SELECT push_enabled, booking_updates, payment_updates, payout_updates, promo_updates, system_updates
     FROM notification_preferences
     WHERE user_id = ?
     LIMIT 1`,
    [userId]
  );

  if (rows.length === 0) {
    return res.json({
      success: true,
      data: {
        push_enabled: 1,
        booking_updates: 1,
        payment_updates: 1,
        payout_updates: 1,
        promo_updates: 1,
        system_updates: 1,
      },
    });
  }

  res.json({
    success: true,
    data: rows[0],
  });
});

/**
 * @desc    Update notification preferences
 * @route   PUT /api/v1/notifications/preferences
 * @access  Private
 */
exports.updateNotificationPreferences = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const payload = req.body || {};
  const toBit = (value) => {
    if (value === true || value === 1 || value === '1') return 1;
    if (typeof value === 'string' && value.toLowerCase() === 'true') return 1;
    return 0;
  };

  const keys = [
    'push_enabled',
    'booking_updates',
    'payment_updates',
    'payout_updates',
    'promo_updates',
    'system_updates',
  ];
  const values = keys.map((key) => toBit(payload[key]));

  await db.query(
    `INSERT INTO notification_preferences (
      user_id, push_enabled, booking_updates, payment_updates, payout_updates, promo_updates, system_updates
     ) VALUES (?, ?, ?, ?, ?, ?, ?)
     ON DUPLICATE KEY UPDATE
      push_enabled = VALUES(push_enabled),
      booking_updates = VALUES(booking_updates),
      payment_updates = VALUES(payment_updates),
      payout_updates = VALUES(payout_updates),
      promo_updates = VALUES(promo_updates),
      system_updates = VALUES(system_updates),
      updated_at = CURRENT_TIMESTAMP`,
    [userId, ...values]
  );

  res.json({
    success: true,
    message: 'Notification preferences updated',
  });
});
