const bcrypt = require('bcrypt');
const db = require('../config/database');
const { generateToken, generateRefreshToken, verifyRefreshToken, generateResetToken, verifyResetToken } = require('../config/jwt');
const { asyncHandler } = require('../middleware/errorHandler');
const { sendEmail, sendSms } = require('../services/notificationService');

const normalizePhone = (value) => {
  const cleaned = String(value || '').replace(/\D/g, '');
  if (!cleaned) return '';
  if (cleaned.startsWith('265') && cleaned.length >= 11) return `+${cleaned}`;
  if (cleaned.startsWith('0') && cleaned.length === 10) return `+265${cleaned.slice(1)}`;
  if (cleaned.length === 9) return `+265${cleaned}`;
  return `+${cleaned}`;
};
const MAX_OTP_SEND_TRIALS = 5;
const MAX_OTP_VERIFY_TRIALS = 5;

/**
 * @desc    Register new user (Client or Detailer)
 * @route   POST /api/v1/auth/register
 * @access  Public
 */
exports.register = asyncHandler(async (req, res) => {
  const {
    email: rawEmail,
    password, 
    role, 
    first_name: rawFirstName,
    last_name: rawLastName,
    business_name: rawBusinessName,
    owner_name,
    phone: rawPhone,
    address,
    city,
    state,
    country,
    latitude,
    longitude,
    description,
    phone_verification_id,
  } = req.body;

  const email = String(rawEmail || '').trim().toLowerCase();
  const phone = String(rawPhone || '').trim();
  const normalizedPhone = normalizePhone(phone);
  const first_name = String(rawFirstName || '').trim();
  const last_name = String(rawLastName || '').trim();
  const business_name = String(rawBusinessName || '').trim();

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Require verified phone before account creation
    if (!phone_verification_id) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Please verify your phone number first',
      });
    }

    const [phoneVerificationRows] = await connection.query(
      `SELECT id
       FROM phone_verifications
       WHERE id = ?
         AND phone = ?
         AND verified_at IS NOT NULL
         AND consumed_at IS NULL
         AND expires_at > NOW()
       LIMIT 1`,
      [phone_verification_id, normalizedPhone]
    );

    if (phoneVerificationRows.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Phone verification is invalid or expired. Please verify again.',
      });
    }

    // Check if user exists by email
    const [existingUsers] = await connection.query(
      'SELECT id FROM users WHERE LOWER(email) = LOWER(?)',
      [email]
    );

    if (existingUsers.length > 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Email already registered'
      });
    }

    // Check if phone already exists across both clients and detailers
    const [existingPhones] = await connection.query(
      `SELECT u.id
       FROM users u
       LEFT JOIN clients c ON c.user_id = u.id AND c.deleted_at IS NULL
       LEFT JOIN detailers d ON d.user_id = u.id AND d.deleted_at IS NULL
       WHERE u.deleted_at IS NULL AND (c.phone = ? OR d.phone = ?)
       LIMIT 1`,
      [phone, phone]
    );

    if (existingPhones.length > 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Phone number already registered'
      });
    }

    // Role-specific duplicate name checks
    if (role === 'client') {
      const [existingClientName] = await connection.query(
        `SELECT id
         FROM clients
         WHERE deleted_at IS NULL
           AND LOWER(first_name) = LOWER(?)
           AND LOWER(last_name) = LOWER(?)
         LIMIT 1`,
        [first_name, last_name]
      );

      if (existingClientName.length > 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Client name already exists'
        });
      }
    }

    if (role === 'detailer') {
      const [existingBusinessName] = await connection.query(
        `SELECT id
         FROM detailers
         WHERE deleted_at IS NULL
           AND LOWER(business_name) = LOWER(?)
         LIMIT 1`,
        [business_name]
      );

      if (existingBusinessName.length > 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Business name already exists'
        });
      }
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Insert user
    const [userResult] = await connection.query(
      'INSERT INTO users (email, password, role) VALUES (?, ?, ?)',
      [email, hashedPassword, role]
    );

    const userId = userResult.insertId;

    // Insert role-specific data
    if (role === 'client') {
      await connection.query(
        `INSERT INTO clients (user_id, first_name, last_name, phone, address, city, state, country, latitude, longitude) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [userId, first_name, last_name, phone, address || null, city || null, state || null, country || 'Malawi', latitude || null, longitude || null]
      );
    } else if (role === 'detailer') {
      await connection.query(
        `INSERT INTO detailers (user_id, business_name, owner_name, phone, address, city, state, country, latitude, longitude, description) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [userId, business_name, owner_name, phone, address, city, state || null, country || 'Malawi', latitude, longitude, description || null]
      );
    }

    await connection.query(
      'UPDATE phone_verifications SET consumed_at = NOW() WHERE id = ?',
      [phone_verification_id]
    );

    await connection.commit();

    // Generate tokens
    const accessToken = generateToken({ userId, email, role });
    const refreshToken = generateRefreshToken({ userId });

    res.status(201).json({
      success: true,
      message: 'Registration successful',
      data: {
        user: {
          id: userId,
          email,
          role
        },
        accessToken,
        refreshToken
      }
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Check registration field availability
 * @route   POST /api/v1/auth/check-availability
 * @access  Public
 */
exports.checkAvailability = asyncHandler(async (req, res) => {
  const {
    role,
    email: rawEmail,
    phone: rawPhone,
    first_name: rawFirstName,
    last_name: rawLastName,
    business_name: rawBusinessName,
  } = req.body || {};

  const email = String(rawEmail || '').trim().toLowerCase();
  const phone = String(rawPhone || '').trim();
  const first_name = String(rawFirstName || '').trim();
  const last_name = String(rawLastName || '').trim();
  const business_name = String(rawBusinessName || '').trim();

  const data = {
    email_exists: false,
    phone_exists: false,
    client_name_exists: false,
    business_name_exists: false,
  };

  if (email) {
    const [rows] = await db.query(
      'SELECT id FROM users WHERE deleted_at IS NULL AND LOWER(email) = LOWER(?) LIMIT 1',
      [email]
    );
    data.email_exists = rows.length > 0;
  }

  if (phone) {
    const [rows] = await db.query(
      `SELECT u.id
       FROM users u
       LEFT JOIN clients c ON c.user_id = u.id AND c.deleted_at IS NULL
       LEFT JOIN detailers d ON d.user_id = u.id AND d.deleted_at IS NULL
       WHERE u.deleted_at IS NULL AND (c.phone = ? OR d.phone = ?)
       LIMIT 1`,
      [phone, phone]
    );
    data.phone_exists = rows.length > 0;
  }

  if (role === 'client' && first_name && last_name) {
    const [rows] = await db.query(
      `SELECT id
       FROM clients
       WHERE deleted_at IS NULL
         AND LOWER(first_name) = LOWER(?)
         AND LOWER(last_name) = LOWER(?)
       LIMIT 1`,
      [first_name, last_name]
    );
    data.client_name_exists = rows.length > 0;
  }

  if (role === 'detailer' && business_name) {
    const [rows] = await db.query(
      `SELECT id
       FROM detailers
       WHERE deleted_at IS NULL
         AND LOWER(business_name) = LOWER(?)
       LIMIT 1`,
      [business_name]
    );
    data.business_name_exists = rows.length > 0;
  }

  res.json({
    success: true,
    data,
  });
});

/**
 * @desc    Send registration OTP to phone
 * @route   POST /api/v1/auth/send-registration-otp
 * @access  Public
 */
exports.sendRegistrationOtp = asyncHandler(async (req, res) => {
  const phone = String(req.body?.phone || '').trim();
  const normalizedPhone = normalizePhone(phone);

  if (!normalizedPhone) {
    return res.status(400).json({
      success: false,
      message: 'Phone number is required',
    });
  }

  // Rate limit: at most 5 OTP send attempts in 10 minutes per phone
  const [recentRows] = await db.query(
    `SELECT COUNT(*) AS total
     FROM phone_verifications
     WHERE phone = ?
       AND created_at > DATE_SUB(NOW(), INTERVAL 10 MINUTE)
       AND consumed_at IS NULL`,
    [normalizedPhone]
  );

  if (Number(recentRows?.[0]?.total || 0) >= MAX_OTP_SEND_TRIALS) {
    return res.status(429).json({
      success: false,
      message: `Too many attempts. You can only send OTP ${MAX_OTP_SEND_TRIALS} times within 10 minutes. Please try again later.`,
    });
  }

  const otp = Math.floor(100000 + Math.random() * 900000).toString();
  const otpHash = await bcrypt.hash(otp, 10);
  const expiresAt = new Date(Date.now() + 10 * 60 * 1000);

  const [insertResult] = await db.query(
    `INSERT INTO phone_verifications (phone, otp_hash, expires_at)
     VALUES (?, ?, ?)`,
    [normalizedPhone, otpHash, expiresAt]
  );

  const message = `Your Simbi Services sign-up OTP is ${otp}. It expires in 10 minutes.`;
  await sendSms({
    to: normalizedPhone,
    message,
  });

  res.json({
    success: true,
    message: 'OTP sent to your phone',
    data: {
      verification_id: insertResult.insertId,
      expires_in_seconds: 600,
    },
  });
});

/**
 * @desc    Verify registration OTP for phone
 * @route   POST /api/v1/auth/verify-registration-otp
 * @access  Public
 */
exports.verifyRegistrationOtp = asyncHandler(async (req, res) => {
  const verificationId = Number(req.body?.verification_id);
  const phone = String(req.body?.phone || '').trim();
  const normalizedPhone = normalizePhone(phone);
  const otp = String(req.body?.otp || '').trim();

  if (!verificationId || !normalizedPhone || !otp) {
    return res.status(400).json({
      success: false,
      message: 'verification_id, phone, and otp are required',
    });
  }

  const [rows] = await db.query(
    `SELECT id, otp_hash, expires_at, verified_at, consumed_at, attempts_count
     FROM phone_verifications
     WHERE id = ? AND phone = ?
     LIMIT 1`,
    [verificationId, normalizedPhone]
  );

  if (rows.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Verification request not found',
    });
  }

  const verification = rows[0];
  if (verification.consumed_at) {
    return res.status(400).json({
      success: false,
      message: 'Verification request already used',
    });
  }
  if (new Date(verification.expires_at).getTime() < Date.now()) {
    return res.status(400).json({
      success: false,
      message: 'OTP expired. Please request a new one.',
    });
  }

  const currentAttempts = Number(verification.attempts_count || 0);
  if (currentAttempts >= MAX_OTP_VERIFY_TRIALS) {
    return res.status(429).json({
      success: false,
      message: `Too many attempts. You only have ${MAX_OTP_VERIFY_TRIALS} OTP verification trials. Please request a new OTP.`,
    });
  }

  const isValidOtp = await bcrypt.compare(otp, verification.otp_hash);
  if (!isValidOtp) {
    const nextAttempts = currentAttempts + 1;
    await db.query(
      'UPDATE phone_verifications SET attempts_count = ? WHERE id = ?',
      [nextAttempts, verificationId]
    );

    if (nextAttempts >= MAX_OTP_VERIFY_TRIALS) {
      return res.status(429).json({
        success: false,
        message: `Too many attempts. You only have ${MAX_OTP_VERIFY_TRIALS} OTP verification trials. Please request a new OTP.`,
      });
    }

    return res.status(400).json({
      success: false,
      message: `Invalid OTP. You have ${MAX_OTP_VERIFY_TRIALS - nextAttempts} trial(s) left.`,
    });
  }

  if (!verification.verified_at) {
    await db.query(
      'UPDATE phone_verifications SET verified_at = NOW() WHERE id = ?',
      [verificationId]
    );
  }

  res.json({
    success: true,
    message: 'Phone number verified successfully',
    data: {
      verification_id: verificationId,
      phone: normalizedPhone,
    },
  });
});

/**
 * @desc    Login user
 * @route   POST /api/v1/auth/login
 * @access  Public
 */
exports.login = asyncHandler(async (req, res) => {
  const { password } = req.body;
  const rawIdentifier = req.body?.identifier || req.body?.email || '';
  const identifier = String(rawIdentifier).trim();
  const normalizedPhone = normalizePhone(identifier);
  const localPhone = normalizedPhone.startsWith('+265') && normalizedPhone.length >= 13
    ? `0${normalizedPhone.substring(4)}`
    : identifier;

  // Get user by email OR phone
  const [users] = await db.query(
    `SELECT u.id, u.email, u.password, u.role, u.is_active
     FROM users u
     LEFT JOIN clients c ON u.id = c.user_id AND c.deleted_at IS NULL
     LEFT JOIN detailers d ON u.id = d.user_id AND d.deleted_at IS NULL
     WHERE u.deleted_at IS NULL
       AND (
         LOWER(u.email) = LOWER(?)
         OR c.phone IN (?, ?, ?)
         OR d.phone IN (?, ?, ?)
       )
     LIMIT 1`,
    [
      identifier,
      identifier,
      normalizedPhone,
      localPhone,
      identifier,
      normalizedPhone,
      localPhone,
    ]
  );

  if (users.length === 0) {
    return res.status(401).json({
      success: false,
      message: 'Invalid email/phone or password'
    });
  }

  const user = users[0];

  // Check if account is active
  if (!user.is_active) {
    return res.status(403).json({
      success: false,
      message: 'Your account has been deactivated. Please contact support.'
    });
  }

  // Verify password
  const isPasswordValid = await bcrypt.compare(password, user.password);

  if (!isPasswordValid) {
    return res.status(401).json({
      success: false,
      message: 'Invalid email/phone or password'
    });
  }

  // Generate tokens
  const accessToken = generateToken({ 
    userId: user.id, 
    email: user.email, 
    role: user.role 
  });
  const refreshToken = generateRefreshToken({ userId: user.id });

  // Get full profile based on role
  let userProfile = {
    id: user.id,
    email: user.email,
    role: user.role
  };

  if (user.role === 'client') {
    const [clients] = await db.query(
      'SELECT first_name, last_name, phone, address, city, profile_image FROM clients WHERE user_id = ?',
      [user.id]
    );
    if (clients.length > 0) {
      userProfile = { ...userProfile, ...clients[0] };
    }
  } else if (user.role === 'detailer') {
    const [detailers] = await db.query(
      'SELECT id as detailer_id, business_name, owner_name, phone, address, city, description, profile_image, is_verified, is_available, rating_average, total_reviews FROM detailers WHERE user_id = ?',
      [user.id]
    );
    if (detailers.length > 0) {
      userProfile = { ...userProfile, ...detailers[0] };
    }
  }

  res.json({
    success: true,
    message: 'Login successful',
    data: {
      user: userProfile,
      accessToken,
      refreshToken
    }
  });
});

/**
 * @desc    Refresh access token
 * @route   POST /api/v1/auth/refresh-token
 * @access  Public
 */
exports.refreshToken = asyncHandler(async (req, res) => {
  const { refreshToken } = req.body;

  if (!refreshToken) {
    return res.status(400).json({
      success: false,
      message: 'Refresh token is required'
    });
  }

  try {
    const decoded = verifyRefreshToken(refreshToken);

    // Get user
    const [users] = await db.query(
      'SELECT id, email, role, is_active FROM users WHERE id = ? AND deleted_at IS NULL',
      [decoded.userId]
    );

    if (users.length === 0 || !users[0].is_active) {
      return res.status(401).json({
        success: false,
        message: 'Invalid refresh token'
      });
    }

    const user = users[0];

    // Generate new access token
    const accessToken = generateToken({ 
      userId: user.id, 
      email: user.email, 
      role: user.role 
    });

    res.json({
      success: true,
      data: { accessToken }
    });

  } catch (error) {
    return res.status(401).json({
      success: false,
      message: 'Invalid or expired refresh token'
    });
  }
});

/**
 * @desc    Request password reset (send OTP)
 * @route   POST /api/v1/auth/forgot-password
 * @access  Public
 */
exports.forgotPassword = asyncHandler(async (req, res) => {
  const { channel, identifier } = req.body;

  const [users] = await db.query(
    `SELECT u.id, u.email, c.phone AS client_phone, d.phone AS detailer_phone
     FROM users u
     LEFT JOIN clients c ON u.id = c.user_id
     LEFT JOIN detailers d ON u.id = d.user_id
     WHERE u.email = ? OR c.phone = ? OR d.phone = ?
     LIMIT 1`,
    [identifier, identifier, identifier]
  );

  if (users.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'User not found'
    });
  }

  const user = users[0];
  const destination = channel === 'email'
    ? user.email
    : (user.client_phone || user.detailer_phone);

  if (!destination) {
    return res.status(400).json({
      success: false,
      message: `No ${channel === 'email' ? 'email' : 'phone'} found for this account`
    });
  }

  // Generate OTP
  const otp = Math.floor(100000 + Math.random() * 900000).toString();
  const otpHash = await bcrypt.hash(otp, 10);
  const expiresAt = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

  await db.query(
    'INSERT INTO password_resets (user_id, channel, otp_hash, expires_at) VALUES (?, ?, ?, ?)',
    [user.id, channel, otpHash, expiresAt]
  );

  const message = `Your Simbi Services OTP is ${otp}. It expires in 10 minutes.`;

  try {
    if (channel === 'email') {
      console.log(`[FORGOT PASSWORD] Sending email to: ${destination}`);
      await sendEmail({
        to: destination,
        subject: 'Your Password Reset OTP',
        text: message
      });
      console.log(`[FORGOT PASSWORD] Email sent successfully`);
    } else {
      console.log(`[FORGOT PASSWORD] Sending SMS to: ${destination}`);
      await sendSms({
        to: destination,
        message
      });
      console.log(`[FORGOT PASSWORD] SMS sent successfully`);
    }
  } catch (error) {
    console.error(`[FORGOT PASSWORD] Error sending ${channel}:`, error.message);
    if (error.response) {
      console.error(`[FORGOT PASSWORD] API Response:`, error.response.data);
    }
    return res.status(500).json({
      success: false,
      message: `Failed to send OTP via ${channel}: ${error.message}`
    });
  }

  res.json({
    success: true,
    message: `OTP sent via ${channel}`
  });
});

/**
 * @desc    Verify OTP and issue reset token
 * @route   POST /api/v1/auth/verify-otp
 * @access  Public
 */
exports.verifyOtp = asyncHandler(async (req, res) => {
  const { channel, identifier, otp } = req.body;

  const [users] = await db.query(
    `SELECT u.id, u.email, c.phone AS client_phone, d.phone AS detailer_phone
     FROM users u
     LEFT JOIN clients c ON u.id = c.user_id
     LEFT JOIN detailers d ON u.id = d.user_id
     WHERE u.email = ? OR c.phone = ? OR d.phone = ?
     LIMIT 1`,
    [identifier, identifier, identifier]
  );

  if (users.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'User not found'
    });
  }

  const user = users[0];

  const [resets] = await db.query(
    `SELECT id, otp_hash, expires_at, used_at
     FROM password_resets
     WHERE user_id = ? AND channel = ?
     ORDER BY created_at DESC
     LIMIT 1`,
    [user.id, channel]
  );

  if (resets.length === 0) {
    return res.status(400).json({
      success: false,
      message: 'OTP not found'
    });
  }

  const reset = resets[0];

  if (reset.used_at) {
    return res.status(400).json({
      success: false,
      message: 'OTP already used'
    });
  }

  if (new Date(reset.expires_at).getTime() < Date.now()) {
    return res.status(400).json({
      success: false,
      message: 'OTP expired'
    });
  }

  const isValid = await bcrypt.compare(otp, reset.otp_hash);
  if (!isValid) {
    return res.status(400).json({
      success: false,
      message: 'Invalid OTP'
    });
  }

  await db.query('UPDATE password_resets SET used_at = NOW() WHERE id = ?', [reset.id]);

  const resetToken = generateResetToken({ userId: user.id, purpose: 'password-reset' });

  res.json({
    success: true,
    message: 'OTP verified',
    data: { resetToken }
  });
});

/**
 * @desc    Reset password using reset token
 * @route   POST /api/v1/auth/reset-password
 * @access  Public
 */
exports.resetPassword = asyncHandler(async (req, res) => {
  const { resetToken, newPassword } = req.body;

  const decoded = verifyResetToken(resetToken);
  if (!decoded || decoded.purpose !== 'password-reset') {
    return res.status(400).json({
      success: false,
      message: 'Invalid reset token'
    });
  }

  const hashedPassword = await bcrypt.hash(newPassword, 10);

  await db.query(
    'UPDATE users SET password = ? WHERE id = ? AND deleted_at IS NULL',
    [hashedPassword, decoded.userId]
  );

  res.json({
    success: true,
    message: 'Password reset successful'
  });
});

/**
 * @desc    Get current user profile
 * @route   GET /api/v1/auth/me
 * @access  Private
 */
exports.getProfile = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const role = req.user.role;

  let query = 'SELECT u.id, u.email, u.role, u.is_active, u.created_at ';
  let joins = 'FROM users u ';

  if (role === 'client') {
    query += ', c.first_name, c.last_name, c.phone, c.address, c.city, c.state, c.country, c.latitude, c.longitude, c.profile_image ';
    joins += 'LEFT JOIN clients c ON u.id = c.user_id ';
  } else if (role === 'detailer') {
    query += ', d.business_name, d.owner_name, d.phone, d.address, d.city, d.state, d.country, d.latitude, d.longitude, d.description, d.profile_image, d.is_verified, d.is_available, d.rating_average, d.total_reviews ';
    joins += 'LEFT JOIN detailers d ON u.id = d.user_id ';
  }

  query += joins + 'WHERE u.id = ? AND u.deleted_at IS NULL';

  const [users] = await db.query(query, [userId]);

  if (users.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'User not found'
    });
  }

  res.json({
    success: true,
    data: users[0]
  });
});

/**
 * @desc    Change password
 * @route   PUT /api/v1/auth/change-password
 * @access  Private
 */
exports.changePassword = asyncHandler(async (req, res) => {
  const { currentPassword, newPassword } = req.body;
  const userId = req.user.id;

  if (!currentPassword || !newPassword) {
    return res.status(400).json({
      success: false,
      message: 'Current password and new password are required'
    });
  }

  if (newPassword.length < 6) {
    return res.status(400).json({
      success: false,
      message: 'New password must be at least 6 characters'
    });
  }

  // Get current password
  const [users] = await db.query(
    'SELECT password FROM users WHERE id = ?',
    [userId]
  );

  if (users.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'User not found'
    });
  }

  // Verify current password
  const isPasswordValid = await bcrypt.compare(currentPassword, users[0].password);

  if (!isPasswordValid) {
    return res.status(401).json({
      success: false,
      message: 'Current password is incorrect'
    });
  }

  // Hash new password
  const hashedPassword = await bcrypt.hash(newPassword, 10);

  // Update password
  await db.query(
    'UPDATE users SET password = ? WHERE id = ?',
    [hashedPassword, userId]
  );

  res.json({
    success: true,
    message: 'Password changed successfully'
  });
});

/**
 * @desc    Logout user
 * @route   POST /api/v1/auth/logout
 * @access  Private
 */
exports.logout = asyncHandler(async (req, res) => {
  // In a production app, you might want to blacklist the token
  res.json({
    success: true,
    message: 'Logged out successfully'
  });
});
